/*:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
::
::  Module      :   Injector Current Select Resource Framework API Header File
::  Copyright   :   (C)2003-2009 Woodward
::  Platform(s) :   MPC5xx
::  Limitations :   None
::
:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::*/
/*! \file Resource_InjCurrSelect.h
    \brief Some of the injector drivers on Woodward's MPC5xx modules are of the switched mode type which control
           to a current level. On these drivers, some allow that current level to be tuned at runtime by the
           application. The BEHAVIOUR_INJCURR_SELECT describes this behaviour. See \ref injcurrentoverview and
           \ref injcurrusage. */
            
#ifndef __RESOURCE_INJCURRSELECT_H
#define __RESOURCE_INJCURRSELECT_H

/*----- INCLUDES ------------------------------------------------------------------------------------------*/
#include <typedefn.h>
#include <resource.h>
#include <NativeError.h>

/*----- DEFINES -------------------------------------------------------------------------------------------*/

/*----- TYPEDEFS ------------------------------------------------------------------------------------------*/
#pragma pack(1)

/*! \brief Enumeration describes the set of mask definitions that are used to identify the valid injector
           current selector attributes in the S_InjCurrResourceAttributes and S_InjCurrCreateResourceAttributes
           data structures.

    Each mask definition applies to a particular member of one of the data structures.
    \code
    ...
    S_InjCurrCreateResourceAttributes CreateObj;

    // Will use DynamicObj
    CreateObj.uValidAttributesMask = USE_INJCURR_DYNAMIC_ON_CREATE;
    // Only going to set the select member
    CreateObj.DynamicObj.uValidAttributesMask = USE_INJCURR_SELECT;
    // The remainder of the data structure can be left undefined since the uValidAttributesMask
    //   indicates that the members are not valid
    ... \endcode */
typedef enum
{
    USE_INJCURR_DEFAULTS = 0,
    USE_INJCURR_SELECT = 0x01,              /*!< Selects S_InjCurrResourceAttributes::eCurrLevel */
    USE_INJCURR_DYNAMIC_ON_CREATE = 0x02,   /*!< Selects S_InjCurrCreateResourceAttributes::DynamicObj */

/* IF THIS TYPE EXCEEDS 0x80 THEN ALTER THE TYPE OF THE uValidAttributesMask MEMBER(S) ACCORDINGLY */

} E_InjCurrAttributesMask;

/*! Enumeration that describes the current selection modes available */
/* Name: E_InjCurrLevel ClassID:ENUMDEC EnumDflt:"Illegal" */
typedef enum
{
/*! The higher current selection */
    INJ_HIGH_CURRENT,    /* EnumTxt:"High Current" */
/*! The lower current selection */
    INJ_LOW_CURRENT      /* EnumTxt:"Low Current" */
} E_InjCurrLevel;

/*! \brief This data structure describes all of the injector current selector resource's runtime configuration attributes.

    The attributes are altered through the use of SetResourceAttributesBEHAVIOUR_INJCURR_SELECT(). The data
    structure does not need to be completely filled inorder to be used. The \b uValidAttributesMask is a bit
    field member that is used to identify which attributes are valid. Each attribute is identified with a
    separate bit mask that is logic-ORed into the a mask when an attribute is to be used.
    \code
    ...
    S_InjCurrResourceAttributes InCurrObj;

    // Only going to set the injector current threshold enumeration member
    InCurrObj.uValidAttributesMask = USE_INJCURR_SELECT;  \endcode */    
typedef struct
{
/*! Logic-OR the attributes [\ref USE_INJCURR_SELECT] that are valid for this this instance of the data structure.
    See \ref injcurrusage for examples on this member's use. */
    uint1 uValidAttributesMask;
/*! Broadly defines the injector current levels to be employed by the associated injector drivers via an enumeration.
    See \ref injcurrusage. Select this attribute with the \ref USE_INJCURR_SELECT. Defaults to \ref INJ_HIGH_CURRENT
    upon resource creation. */
    E_InjCurrLevel eCurrLevel;
} S_InjCurrResourceAttributes;

typedef S_InjCurrResourceAttributes const* S_InjCurrResourceAttributesPtr;

/*! \brief This data structure describes the creation attributes for an injector driver current level selection resource

    Included within the structure is the \b DynamicObj which represents the initial states for the resource's
    dynamic attributes. This data structure is referenced when the resource is created via the
    CreateResourceBEHAVIOUR_INJCURR_SELECT() function and \ref USE_INJCURR_DYNAMIC_ON_CREATE is included in the
    S_InjCurrCreateResourceAttributes::uValidAttributesMask member.
    
    \code
    ...
    S_InjCurrCreateResourceAttributes CreateObj;

    // Will use DynamicObj
    CreateObj.uValidAttributesMask = USE_INJCURR_DYNAMIC_ON_CREATE;
    // Only going to set the select member
    CreateObj.DynamicObj.uValidAttributesMask = USE_INJCURR_SELECT;
    // The rest of the data structure can remain uninitialised because the framework will ignore it \endcode */    
typedef struct
{
/*! Logic-OR the attributes [\ref USE_CJ125_LSU_DYNAMIC_ON_CREATE] that are valid for this this instance of the data structure.

    \code
    ...
    S_InjCurrCreateResourceAttributes CreateObj;

    // Will use DynamicObj
    CreateObj.uValidAttributesMask = USE_INJCURR_DYNAMIC_ON_CREATE;
    // Only going to set the select member
    CreateObj.DynamicObj.uValidAttributesMask = USE_INJCURR_SELECT;
    // The rest of the data structure can remain uninitialised because the framework will ignore it \endcode */    
    uint1 uValidAttributesMask;
/*! Initial values of the runtime attributes, which can be later altered through a call to
    SetResourceAttributesBEHAVIOUR_INJCURR_SELECT(). Select with the \ref USE_INJCURR_DYNAMIC_ON_CREATE bit mask */
    S_InjCurrResourceAttributes DynamicObj;
} S_InjCurrCreateResourceAttributes;

typedef S_InjCurrCreateResourceAttributes const* S_InjCurrCreateResourceAttributesPtr;

#pragma pack()
/*----- EXTERNALS -----------------------------------------------------------------------------------------*/

/*----- PROTOTYPES ----------------------------------------------------------------------------------------*/
NativeError_S CreateResourceBEHAVIOUR_INJCURR_SELECT(E_ModuleResource, S_InjCurrCreateResourceAttributes const*);
NativeError_S SetResourceAttributesBEHAVIOUR_INJCURR_SELECT(E_ModuleResource, S_InjCurrResourceAttributes const*);

#endif /* __RESOURCE_INJCURRSELECT_H */

/*----- END OF FILE ---------------------------------------------------------------------------------------*/
